<?php
/**
 * Integrator
 * WHMCS - API Task Handler
 * 
 * @package    Integrator
 * @copyright  2009 - 2014 Go Higher Information Services.  All rights reserved.
 * @license    Commercial
 * @version    3.0.21 ( $Id: integrator.php 309 2014-01-14 16:46:18Z steven_gohigher $ )
 * @author     Go Higher Information Services
 * @since      3.0.0
 * 
 * @desc       This file handles API calls from the Integrator application to WHMCS where custom tasks must be performed
 * 
 */

/*-- Security Protocols --*/
defined( 'WHMCS' ) or die( 'Restricted access' );
/*-- Security Protocols --*/

// Be sure a task is set
if (! isset ( $task ) ) {
	$task = 'info';
}

// Create object
$iapi = new IntegratorApi();
$iapi->$task();

/**
 * Integrator API class
 * @version		3.0.21
 * 
 * @since		3.0.0
 * @author		Steven
 */
class IntegratorApi
{
	/**
	 * Contains the version of this file
	 * @access		public
	 * @version		3.0.21
	 * @var			string
	 */
	public $version 	= '3.0.21';
	
	/**
	 * Stores the character set of the system
	 * @access		public
	 * @version		3.0.21
	 * @var			string
	 */
	public $charset		= "UTF-8";
	
	/**
	 * Stores whether values should be converted
	 * @access		public
	 * @since		3.0.0
	 * @var			boolean
	 */
	public $iconv		= false;
	
	
	/**
	 * Stores the WHMCS version of this system
	 * @access		public
	 * @since		3.0.0 (0.3)
	 * @var			integer
	 */
	public $WHMCSversion	= 0;
	
	
	/**
	 * Constructor method
	 * @access		public
	 * @version		3.0.21
	 * 
	 * @since		3.0.0
	 */
	public function __construct()
	{
		$this->_set_characterset();
		$this->_set_version();
	}
	
	
	/**
	 * Assembles the available languages for this connection
	 * @access		public
	 * @version		3.0.21
	 * 
	 * @since		3.0.0
	 */
	public function get_languages()
	{
		if (! defined ( "ROOTDIR" ) ) {
			define( "ROOTDIR", dirname(dirname(dirname(__FILE__ ) ) ) );
		}
		
		$rows	= array();
		$d = dir ( ROOTDIR . DIRECTORY_SEPARATOR . "lang" );
		
		while ( false !== ( $file = $d->read() ) ) {
			if ( in_array( $file, array( ".", ".." ) ) ) continue;
			if (is_dir( ROOTDIR . DIRECTORY_SEPARATOR . "lang" . DIRECTORY_SEPARATOR . $file ) ) continue;
			if ( strpos( $file, ".php" ) === false ) continue;
			$file = substr( $file, 0, -4 );
			$rows[$file] = $file;
		}
		
		$d->close();
		$this->_close( $rows );
	}
	
	
	/**
	 * Assembles the available pages for this connection
	 * @access		public
	 * @version		3.0.21
	 * 
	 * @since		3.0.0
	 */
	public function get_pages()
	{
		if (! defined ( "ROOTDIR" ) ) {
			define( "ROOTDIR", dirname(dirname(dirname(__FILE__ ) ) ) );
		}
		
		$rows = array(
			'clientarea'		=> 'Client Area',			'index'				=> 'Client Portal',			'clientarea-x-invoices'			=> 'View Invoices',			'clientarea-x-details'			=> 'My Details',			'clientarea-x-products'			=> 'My Products',			'clientarea-x-domains'			=> 'My Domains',			'clientarea-x-quotes'			=> 'My Quotes',			'clientarea-x-emails'			=> 'My Emails',			'cart-a-add-domain-transfer'	=>	'Cart: Add with Domain Transfer',			'cart'							=>	'Cart: Shopping Cart',			'cart-a-confproduct'			=>	'Cart: Configure Product',			'cart-a-view'					=>	'Cart: View Cart',			'cart-gid-renewals'				=>	'Cart: Domain Renewals',			'supporttickets'				=> 'Support Tickets',			'affiliates'					=> 'Affiliate',			'announcements'					=> 'Announcements',			'banned'						=> 'Banned',			'configuressl'					=> 'Configure SSL',			'contact'						=> 'Contact',			'creditcard'					=> 'Credit Card',			'domainchecker'					=> 'Domain Checker',			'downloads'						=> 'Downloads',			'knowledgebase'					=> 'Knowledgebase',			'networkissues'					=> 'Network Issues',			'pwreset'						=> 'Password Reset',			'pwresetvalidation'				=> 'Password Reset Validation',			'register'						=> 'Register',			'serverstatus'						=> 'Server Status',			'submitticket'						=> 'Submit Ticket',			'tutorials'							=> 'Tutorials',			'domainchecker-search-bulkregister'	=>	'Domain Checker: Bulk Register',			'domainchecker-search-bulktransfer'	=>	'Domain Checker: Bulk Transfer',
		);
		
		if ( $this->WHMCSversion < 500 ) {
			$rows['order']	= 'Order';
		}
		
		ksort( $rows );
		$this->_close( $rows );
	}
	
	
	/**
	 * Retrieves the current version of this addon module
	 * @access		public
	 * @version		3.0.21
	 * 
	 * @since		3.0.1 (0.1)
	 */
	public function get_version()
	{
		$rows['cnxns|whmcs'] = $this->version;
		$this->_close( $rows );
	}
	
	
	/**
	 * Sends back information about this connection
	 * @access		public
	 * @version		3.0.21
	 * 
	 * @since		3.0.0
	 */
	public function info()
	{
		$rows['version'] = $this->version;
		$this->_close( $rows );
	}
	
	
	/**
	 * Tests connection is active
	 * @access		public
	 * @version		3.0.21
	 * 
	 * @since		3.0.0
	 */
	public function ping()
	{
		$rows['data'] = "pong";
		$this->_close( $rows );
	}
	
	
	/**
	 * Update settings in I3
	 * @access		public
	 * @version		3.0.21
	 * 
	 * @since		3.0.0 (0.2)
	 */
	public function update_settings()
	{
		if (! defined ( "ROOTDIR" ) ) {
			define( "ROOTDIR", dirname(dirname(dirname(__FILE__ ) ) ) );
		}
		
		global $settings;
		
		$config	= & IntFactory :: getConfig();
		$check	=   array( 'debug' => array( 'Debug', array( '0' => 'No', '1' => 'Yes' ) ) );
		$bind	=   array();
		
		foreach ( $check as $chk => $b ) {
			if ( isset( $settings[$chk] ) ) {
				$bind[$b[0]] = $b[1][$settings[$chk]];
			}
		}
		
		if ( empty( $bind ) ) {
			return $this->_close( array( 'result' => 'error', 'data' => 'No settings sent to update' ) );
		}
		
		$config->bind( $bind );
		$config->save();
		
		return $this->_close( array( 'result' => 'success', 'data' => 'Settings updated!' ) );
	}
	
	
	/**
	 * User search functionality - b/c WHMCS doesn't search properly
	 * @access		public
	 * @version		3.0.21
	 *
	 * @since		3.0.20
	 */
	public function user_search()
	{
		global $search, $limitstart, $limitnum;
		
		if (! $limitnum ) $limitnum = '25';
		if (! $limitstart && $limitstart != '0' ) $limitstart = '0';
		
		$db		=	IntFactory :: getDbo();
		$query	=	"SELECT SQL_CALC_FOUND_ROWS id, firstname, lastname, companyname, email, groupid, datecreated, status FROM tblclients WHERE email LIKE '%" . $search . "%' OR firstname LIKE '%" . $search . "%' OR lastname LIKE '%" . $search . "%' OR companyname LIKE '%" . $search . "%' ORDER BY firstname, lastname, companyname LIMIT " . (int)$limitstart . ", " . (int)$limitnum;
		$db->setQuery( $query );
		$result	=	$db->loadObjectList();
		
		$db->setQuery( "SELECT FOUND_ROWS()" );
		$total	=	$db->loadResult();
		
		$apiresults	=	array(
			"totalresults"	=>	$total,
			"startnumber"	=>	$limitstart,
			"numreturned"	=>	count( $result )
		);
		
		foreach ( $result as $item ) {
			$apiresults["clients"]["client"][] = array(
				"id"			=>	$item->id,
				"firstname"		=>	$item->firstname,
				"lastname"		=>	$item->lastname,
				"companyname"	=>	$item->companyname,
				"email"			=>	$item->email,
				"datecreated"	=>	$item->datecreated,
				"groupid"		=>	$item->groupid,
				"status"		=>	$item->status
			);
		}
		
		$this->_close( $apiresults );
	}
	
	
	/**
	 * Validates a password
	 * @access		public
	 * @version		3.0.21
	 * 
	 * @since		3.0.0
	 */
	public function validate_password()
	{
		global $password2;
		
		if ( ( $password2 == null ) || ( $password2 == '' ) ) {
			return $this->_close( array( 'result' => 'error', 'data' => 'No password sent or found' ) );
		}
		
		$pwstrength	= $this->_get_setting( 'RequiredPWStrength' );
		$sentpw		= $this->_calculate_password_strength( $password2 );
		
		if ( $sentpw < $pwstrength ) {
			return $this->_close( array( 'result' => 'error', 'data' => sprintf( 'Password not strong enough ( %s < %s )', $sentpw, $pwstrength ) ) );
		}
		
		return $this->_close( array( 'result' => 'success', 'data' => sprintf( 'Password strong enough! ( %s >= %s )', $sentpw, $pwstrength ) ) );	
	}
	
	
	/**
	 * Closes the application
	 * @access		private
	 * @version		3.0.21
	 * @param		mixed		- $data: the data to send back to the calling application
	 * @param		boolean		- $first: if true this is the first run (allows recursion)
	 * 
	 * @since		3.0.0
	 */
	private function _close( $data, $first = true )
	{
		if ( $this->iconv ) {
			if (! defined ( "ROOTDIR" ) ) {
				define( "ROOTDIR", dirname(dirname(dirname(__FILE__ ) ) ) );
			}
			$filename	= ROOTDIR . DIRECTORY_SEPARATOR . 'modules' . DIRECTORY_SEPARATOR . 'addons' . DIRECTORY_SEPARATOR . 'integrator' . DIRECTORY_SEPARATOR . 'helpers' . DIRECTORY_SEPARATOR . 'language.php';
			
			if ( file_exists( $filename ) ) {
				require_once( $filename );
				$data = my_iconv( $data, $this->charset, "UTF-8" );
			}
		}
		
		if (! isset( $data['result'] ) ) $data['result'] = 'success';
		echo json_encode( $data ); 
	}
	
	
	/**
	 * Calculates the password strength of a given password
	 * @access		private
	 * @version		3.0.21
	 * @param		string		- $pw: contains the new password
	 * 
	 * @return		integer containing the calculated strength
	 * @since		3.0.0
	 */
	private function _calculate_password_strength( $pw )
	{
		// String Length
		$pwlength = strlen($pw);
		if ( $pwlength > 5 ) $pwlength = 5;
		
		// How many numbers
		$numnumeric = preg_replace( "/[0-9]/", "", $pw );
		$numeric = strlen( $pw ) - strlen( $numnumeric );
		if ( $numeric > 3 ) $numeric = 3;
		
		// How many symbols
		$symbols = preg_replace( "/\W/", "", $pw );
		$numsymbols = strlen( $pw ) - strlen( $symbols );
		if ( $numsymbols > 3 ) $numsymbols = 3;
		
		// How many uppercase
		$numupper = preg_replace( "/[A-Z]/", "", $pw );
		$upper = strlen( $pw ) - strlen( $numupper );
		if ( $upper > 3 ) $upper = 3;
		
		// Calculate password strength
		$pwstrength = ( ( $pwlength * 10 ) - 20 ) + ( $numeric * 10 ) + ( $numsymbols * 15 ) + ( $upper * 10 );
		
		// Keep strength between 0 and 100 and return
		if ( $pwstrength < 0 ) $pwstrength = 0;
		if ( $pwstrength > 100 ) $pwstrength = 100;
		return $pwstrength;
	}
	
	
	/**
	 * Retrieves a setting from the WHMCS configuration table
	 * @access		private
	 * @version		3.0.21
	 * @param		string		- $key: contains the table key to find
	 * 
	 * @return		mixed value depending on what is found
	 * @since		3.0.0
	 */
	private function _get_setting( $key )
	{
		$query	= "SELECT `value` FROM `tblconfiguration` WHERE `setting` = '{$key}'";
		$result	= mysql_query( $query );
		while ($row = mysql_fetch_assoc( $result ) ) {
			$item = $row['value'];
		}
		return $item;
	}
	
	
	/**
	 * Sets the character set used by WHMCS
	 * @access		private
	 * @version		3.0.21
	 * 
	 * @since		3.0.0
	 */
	private function _set_characterset()
	{
		$charset	= $this->_get_setting( 'Charset' );
		$this->charset = strtoupper( $charset );
		$this->iconv = ( function_exists( "iconv" ) ? ( $this->charset == "UTF-8" ? false : true ) : false );
		
		return;
	}
	
	
	/**
	 * Sets the version of WHMCS internally
	 * @access		private
	 * @version		3.0.21
	 * 
	 * @since		3.0.0 (0.3)
	 */
	private function _set_version()
	{
		$version	= $this->_get_setting( 'Version' );
		$version	= str_replace( '.', '', $version );
		$this->WHMCSversion = (int) $version;
		return;
	}
}
?>